<?php

namespace App\Services;

use App\Models\DeviceToken;
use App\Models\User;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class PushNotificationService
{
    /**
     * Register a device token for push notifications.
     */
    public function registerToken(User $user, string $token, string $platform, ?string $deviceId = null, ?string $deviceName = null): DeviceToken
    {
        return DeviceToken::updateOrCreate(
            [
                'user_id' => $user->id,
                'token' => $token,
            ],
            [
                'platform' => $platform,
                'device_id' => $deviceId,
                'device_name' => $deviceName,
                'is_active' => true,
            ]
        );
    }

    /**
     * Unregister a device token.
     */
    public function unregisterToken(string $token): void
    {
        DeviceToken::where('token', $token)->update(['is_active' => false]);
    }

    /**
     * Send push notification to Android device (FCM).
     */
    public function sendToAndroid(string $token, string $title, string $message, array $data = []): bool
    {
        $fcmServerKey = config('services.fcm.server_key');
        
        if (!$fcmServerKey) {
            Log::warning('FCM server key not configured');
            return false;
        }

        try {
            $response = Http::withHeaders([
                'Authorization' => 'key=' . $fcmServerKey,
                'Content-Type' => 'application/json',
            ])->post('https://fcm.googleapis.com/fcm/send', [
                'to' => $token,
                'notification' => [
                    'title' => $title,
                    'body' => $message,
                    'sound' => 'default',
                    'badge' => 1,
                ],
                'data' => $data,
                'priority' => 'high',
            ]);

            if ($response->successful()) {
                return true;
            }

            Log::error('FCM push notification failed', [
                'status' => $response->status(),
                'body' => $response->body(),
            ]);

            return false;
        } catch (\Exception $e) {
            Log::error('FCM push notification error', [
                'message' => $e->getMessage(),
                'token' => substr($token, 0, 20) . '...',
            ]);

            return false;
        }
    }

    /**
     * Send push notification to iOS device (APNs).
     */
    public function sendToIOS(string $token, string $title, string $message, array $data = []): bool
    {
        // APNs implementation would go here
        // Requires Apple Developer account and APNs certificate/key
        // For now, return false as it requires additional setup
        
        Log::info('iOS push notification not yet implemented. Requires APNs configuration.');
        return false;
    }

    /**
     * Send push notification to user's devices.
     */
    public function sendToUser(User $user, string $title, string $message, array $data = []): void
    {
        if (!$user->canReceiveAutomatedFeatures()) {
            return;
        }

        $tokens = $user->deviceTokens()
            ->where('is_active', true)
            ->get();

        foreach ($tokens as $deviceToken) {
            $sent = false;

            switch ($deviceToken->platform) {
                case 'android':
                    $sent = $this->sendToAndroid($deviceToken->token, $title, $message, $data);
                    break;

                case 'ios':
                    $sent = $this->sendToIOS($deviceToken->token, $title, $message, $data);
                    break;

                case 'web':
                    // Web push notifications handled separately
                    break;
            }

            if ($sent) {
                Log::info("Push notification sent to {$deviceToken->platform} device", [
                    'user_id' => $user->id,
                    'device_id' => $deviceToken->device_id,
                ]);
            }
        }
    }

    /**
     * Send push notification to all user's devices.
     */
    public function sendToAllDevices(User $user, string $title, string $message, array $data = []): void
    {
        $this->sendToUser($user, $title, $message, $data);
    }
}
