<?php

namespace App\Services;

use App\Models\Assignment;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class AssignmentService
{
    /**
     * Create a new assignment with strict validation.
     * Enforces: no assignment without due date and planned study time.
     */
    public function createAssignment(User $user, array $data): Assignment
    {
        $validator = Validator::make($data, [
            'module_name' => 'required|string|max:255',
            'assignment_title' => 'required|string|max:255',
            'tutor_name' => 'required|string|max:255',
            'assignment_type' => 'required|in:weekly,formative,summative',
            'due_date' => 'required|date|after_or_equal:today',
            'planned_study_day' => 'required|date|after_or_equal:today',
            'planned_study_time' => 'required|date_format:H:i',
            'week_start' => 'nullable|integer|min:1|required_if:assignment_type,formative,summative',
            'week_end' => 'nullable|integer|min:1|required_if:assignment_type,formative,summative|gte:week_start',
        ]);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        // Convert planned_study_time to proper format
        $plannedStudyDateTime = Carbon::parse($data['planned_study_day'] . ' ' . $data['planned_study_time']);

        $assignment = Assignment::create([
            'user_id' => $user->id,
            'module_name' => $data['module_name'],
            'assignment_title' => $data['assignment_title'],
            'tutor_name' => $data['tutor_name'],
            'assignment_type' => $data['assignment_type'],
            'due_date' => Carbon::parse($data['due_date']),
            'planned_study_day' => Carbon::parse($data['planned_study_day']),
            'planned_study_time' => $plannedStudyDateTime,
            'week_start' => $data['week_start'] ?? null,
            'week_end' => $data['week_end'] ?? null,
            'status' => 'pending',
        ]);

        return $assignment;
    }

    /**
     * Update an assignment.
     */
    public function updateAssignment(Assignment $assignment, array $data): Assignment
    {
        $validator = Validator::make($data, [
            'module_name' => 'sometimes|required|string|max:255',
            'assignment_title' => 'sometimes|required|string|max:255',
            'tutor_name' => 'sometimes|required|string|max:255',
            'assignment_type' => 'sometimes|required|in:weekly,formative,summative',
            'due_date' => 'sometimes|required|date',
            'planned_study_day' => 'sometimes|required|date',
            'planned_study_time' => 'sometimes|required|date_format:H:i',
            'week_start' => 'nullable|integer|min:1',
            'week_end' => 'nullable|integer|min:1|gte:week_start',
        ]);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        if (isset($data['planned_study_day']) && isset($data['planned_study_time'])) {
            $plannedStudyDateTime = Carbon::parse($data['planned_study_day'] . ' ' . $data['planned_study_time']);
            $data['planned_study_time'] = $plannedStudyDateTime;
        }

        if (isset($data['due_date'])) {
            $data['due_date'] = Carbon::parse($data['due_date']);
        }

        if (isset($data['planned_study_day'])) {
            $data['planned_study_day'] = Carbon::parse($data['planned_study_day']);
        }

        $assignment->update($data);

        return $assignment->fresh();
    }

    /**
     * Mark assignment as completed.
     */
    public function completeAssignment(Assignment $assignment): Assignment
    {
        $assignment->update(['status' => 'completed']);
        return $assignment;
    }

    /**
     * Get assignments for weekly planner.
     */
    public function getWeeklyAssignments(User $user): array
    {
        $startOfWeek = Carbon::now()->startOfWeek();
        $endOfWeek = Carbon::now()->endOfWeek();

        $dueThisWeek = $user->assignments()
            ->pending()
            ->whereBetween('due_date', [$startOfWeek, $endOfWeek])
            ->orderBy('due_date')
            ->get();

        $overdue = $user->assignments()
            ->pending()
            ->where('due_date', '<', Carbon::now())
            ->orderBy('due_date')
            ->get();

        $plannedThisWeek = $user->assignments()
            ->pending()
            ->whereBetween('planned_study_day', [$startOfWeek, $endOfWeek])
            ->orderBy('planned_study_day')
            ->orderBy('planned_study_time')
            ->get();

        return [
            'due_this_week' => $dueThisWeek,
            'overdue' => $overdue,
            'planned_this_week' => $plannedThisWeek,
        ];
    }
}
