<?php

namespace App\Providers;

use App\Jobs\CarryOverJob;
use App\Jobs\EscalationJob;
use App\Jobs\SendStudyReminderJob;
use App\Jobs\WeeklyRiskJob;
use App\Jobs\WhatsAppJob;
use App\Models\Assignment;
use App\Services\NotificationService;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Support\ServiceProvider;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        $this->app->booted(function () {
            $schedule = $this->app->make(Schedule::class);
            
            // Carry over missed study sessions (runs daily at midnight)
            $schedule->call(function () {
                CarryOverJob::dispatch();
            })->daily();

            // Check for overdue assignments and send escalation notifications (runs every 6 hours)
            $schedule->call(function () {
                $overdueAssignments = Assignment::overdue()->get();
                
                foreach ($overdueAssignments as $assignment) {
                    if ($assignment->user->canReceiveAutomatedFeatures()) {
                        EscalationJob::dispatch($assignment);
                    }
                }
            })->everySixHours();

            // Send weekly risk warnings (runs every Sunday at 6 PM)
            $schedule->call(function () {
                WeeklyRiskJob::dispatch();
            })->weeklyOn(0, '18:00');

            // Send study reminders (runs every 15 minutes)
            $schedule->call(function () {
                $upcomingAssignments = Assignment::pending()
                    ->where('planned_study_day', '>=', now()->startOfDay())
                    ->where('planned_study_day', '<=', now()->endOfDay())
                    ->get();

                foreach ($upcomingAssignments as $assignment) {
                    $plannedTime = $assignment->getPlannedStudyDateTime();
                    $reminderTime = $plannedTime->copy()->subMinutes(15);
                    
                    // Send reminder 15 minutes before planned time
                    if (now()->greaterThanOrEqualTo($reminderTime) && now()->lessThan($plannedTime)) {
                        if ($assignment->user->canReceiveAutomatedFeatures()) {
                            SendStudyReminderJob::dispatch($assignment);
                        }
                    }
                }
            })->everyFifteenMinutes();

            // Send due date approaching notifications (runs daily at 8 AM)
            $schedule->call(function () {
                $notificationService = app(NotificationService::class);
                
                $assignmentsDueSoon = Assignment::pending()
                    ->whereBetween('due_date', [now(), now()->addDay()])
                    ->get();

                foreach ($assignmentsDueSoon as $assignment) {
                    if ($assignment->user->canReceiveAutomatedFeatures()) {
                        $notificationService->sendDueDateApproaching($assignment);
                        
                        // Queue WhatsApp notification if phone verified
                        if ($assignment->user->phone) {
                            WhatsAppJob::dispatch($assignment->user, 'due_date_approaching', $assignment);
                        }
                    }
                }
            })->dailyAt('08:00');

            // Send overdue WhatsApp notifications (runs daily at 9 AM)
            $schedule->call(function () {
                $overdueAssignments = Assignment::overdue()->get();
                
                foreach ($overdueAssignments as $assignment) {
                    if ($assignment->user->canReceiveAutomatedFeatures() && $assignment->user->phone) {
                        WhatsAppJob::dispatch($assignment->user, 'overdue', $assignment);
                    }
                }
            })->dailyAt('09:00');
        });
    }
}
