<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class UserPreference extends Model
{
    protected $fillable = [
        'user_id',
        'quiet_hours_start',
        'quiet_hours_end',
        'university',
        'course',
        'year_of_study',
        'study_habits',
        'onboarding_completed',
    ];

    protected function casts(): array
    {
        return [
            'quiet_hours_start' => 'datetime',
            'quiet_hours_end' => 'datetime',
            'onboarding_completed' => 'boolean',
        ];
    }

    /**
     * Get the user that owns the preferences.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Check if current time is within quiet hours.
     */
    public function isQuietHours(): bool
    {
        $now = now();
        $start = $this->quiet_hours_start ?? now()->setTime(18, 0);
        $end = $this->quiet_hours_end ?? now()->setTime(8, 0);

        // Handle quiet hours that span midnight (e.g., 6 PM to 8 AM)
        if ($start->greaterThan($end)) {
            return $now->greaterThanOrEqualTo($start) || $now->lessThanOrEqualTo($end);
        }

        return $now->greaterThanOrEqualTo($start) && $now->lessThanOrEqualTo($end);
    }
}
