<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'trial_start_date',
        'trial_expiry_date',
        'subscription_status',
        'phone',
        'is_admin',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'trial_start_date' => 'date',
            'trial_expiry_date' => 'date',
            'is_admin' => 'boolean',
        ];
    }

    /**
     * Get the user's assignments.
     */
    public function assignments()
    {
        return $this->hasMany(Assignment::class);
    }

    /**
     * Get the user's preferences.
     */
    public function preferences()
    {
        return $this->hasOne(UserPreference::class);
    }

    /**
     * Get the user's notifications.
     */
    public function appNotifications()
    {
        return $this->hasMany(AppNotification::class);
    }

    /**
     * Get the user's device tokens.
     */
    public function deviceTokens()
    {
        return $this->hasMany(DeviceToken::class);
    }

    /**
     * Check if user has active trial or paid subscription.
     */
    public function hasActiveSubscription(): bool
    {
        if ($this->subscription_status === 'paid') {
            return true;
        }

        if ($this->subscription_status === 'trial' && $this->trial_expiry_date) {
            return $this->trial_expiry_date->isFuture();
        }

        return false;
    }

    /**
     * Check if user is in trial period.
     */
    public function isInTrial(): bool
    {
        return $this->subscription_status === 'trial' 
            && $this->trial_expiry_date 
            && $this->trial_expiry_date->isFuture();
    }

    /**
     * Check if user can receive automated features.
     */
    public function canReceiveAutomatedFeatures(): bool
    {
        return $this->hasActiveSubscription();
    }

    /**
     * Check if user is an admin.
     */
    public function isAdmin(): bool
    {
        return (bool) $this->is_admin;
    }
}
