<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class Assignment extends Model
{
    protected $fillable = [
        'user_id',
        'module_name',
        'assignment_title',
        'tutor_name',
        'assignment_type',
        'due_date',
        'planned_study_day',
        'planned_study_time',
        'status',
        'week_start',
        'week_end',
        'carry_over_count',
        'last_carried_over_at',
    ];

    protected function casts(): array
    {
        return [
            'due_date' => 'date',
            'planned_study_day' => 'date',
            'planned_study_time' => 'datetime',
            'last_carried_over_at' => 'datetime',
        ];
    }

    /**
     * Get the user that owns the assignment.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get notifications for this assignment.
     */
    public function notifications()
    {
        return $this->hasMany(AppNotification::class);
    }

    /**
     * Check if assignment is overdue.
     */
    public function isOverdue(): bool
    {
        return $this->status === 'pending' && $this->due_date->isPast();
    }

    /**
     * Check if assignment is due this week.
     */
    public function isDueThisWeek(): bool
    {
        $startOfWeek = Carbon::now()->startOfWeek();
        $endOfWeek = Carbon::now()->endOfWeek();
        
        return $this->due_date->between($startOfWeek, $endOfWeek);
    }

    /**
     * Check if planned study session was missed.
     */
    public function isPlannedStudyMissed(): bool
    {
        if ($this->status === 'completed') {
            return false;
        }

        $plannedDateTime = Carbon::parse($this->planned_study_day->format('Y-m-d') . ' ' . $this->planned_study_time->format('H:i:s'));
        
        return $plannedDateTime->isPast();
    }

    /**
     * Get the planned study datetime.
     */
    public function getPlannedStudyDateTime(): Carbon
    {
        return Carbon::parse($this->planned_study_day->format('Y-m-d') . ' ' . $this->planned_study_time->format('H:i:s'));
    }

    /**
     * Scope: Get pending assignments.
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope: Get completed assignments.
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope: Get overdue assignments.
     */
    public function scopeOverdue($query)
    {
        return $query->pending()->where('due_date', '<', Carbon::now());
    }

    /**
     * Scope: Get assignments due this week.
     */
    public function scopeDueThisWeek($query)
    {
        $startOfWeek = Carbon::now()->startOfWeek();
        $endOfWeek = Carbon::now()->endOfWeek();
        
        return $query->whereBetween('due_date', [$startOfWeek, $endOfWeek]);
    }
}
