<?php

namespace App\Http\Controllers;

use App\Models\UserPreference;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;

class ProfileController extends Controller
{
    /**
     * Show the user profile.
     */
    public function show()
    {
        $user = Auth::user();
        $preferences = $user->preferences;

        return view('profile.show', compact('user', 'preferences'));
    }

    /**
     * Update user profile.
     */
    public function update(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'phone' => ['nullable', 'string', 'max:20', 'unique:users,phone,' . $user->id],
        ]);

        $user->update($request->only(['name', 'email', 'phone']));

        return redirect()
            ->route('profile.show')
            ->with('success', 'Profile updated successfully!');
    }

    /**
     * Update password.
     */
    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => ['required', 'current_password'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
        ]);

        Auth::user()->update([
            'password' => Hash::make($request->password),
        ]);

        return redirect()
            ->route('profile.show')
            ->with('success', 'Password updated successfully!');
    }

    /**
     * Update preferences (quiet hours, etc.).
     */
    public function updatePreferences(Request $request)
    {
        $user = Auth::user();
        
        // Only paid users can customize quiet hours
        if (!$user->canReceiveAutomatedFeatures()) {
            return back()->with('error', 'This feature requires an active subscription.');
        }

        $request->validate([
            'quiet_hours_start' => ['required', 'date_format:H:i'],
            'quiet_hours_end' => ['required', 'date_format:H:i'],
        ]);

        $preferences = $user->preferences ?? new UserPreference(['user_id' => $user->id]);
        $preferences->quiet_hours_start = \Carbon\Carbon::parse($request->quiet_hours_start);
        $preferences->quiet_hours_end = \Carbon\Carbon::parse($request->quiet_hours_end);
        $preferences->save();

        return redirect()
            ->route('profile.show')
            ->with('success', 'Preferences updated successfully!');
    }

    /**
     * Cancel subscription.
     */
    public function cancelSubscription(Request $request)
    {
        $user = Auth::user();
        
        if ($user->subscription_status !== 'paid') {
            return back()->with('error', 'You do not have an active paid subscription.');
        }

        // Change subscription status to free
        $user->update([
            'subscription_status' => 'free',
        ]);

        return redirect()
            ->route('profile.show')
            ->with('success', 'Subscription cancelled successfully. You will retain access until the end of your billing period.');
    }
}
