<?php

namespace App\Http\Controllers;

use App\Models\Assignment;
use App\Services\AssignmentService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;

class AssignmentController extends Controller
{
    public function __construct(
        private AssignmentService $assignmentService
    ) {}

    /**
     * Display a listing of all assignments.
     */
    public function index()
    {
        $user = Auth::user();
        $assignments = $user->assignments()
            ->orderBy('due_date')
            ->paginate(20);

        return view('assignments.index', compact('assignments'));
    }

    /**
     * Show the form for creating a new assignment.
     */
    public function create()
    {
        return view('assignments.create');
    }

    /**
     * Store a newly created assignment.
     * Enforces strict validation: no assignment without due date and planned study time.
     */
    public function store(Request $request)
    {
        try {
            $assignment = $this->assignmentService->createAssignment(Auth::user(), $request->all());

            return redirect()
                ->route('weekly-planner')
                ->with('success', 'Assignment created successfully!');
        } catch (ValidationException $e) {
            return back()
                ->withErrors($e->errors())
                ->withInput();
        }
    }

    /**
     * Display the specified assignment.
     */
    public function show(Assignment $assignment)
    {
        // Ensure user owns this assignment
        if ($assignment->user_id !== Auth::id()) {
            abort(403);
        }

        return view('assignments.show', compact('assignment'));
    }

    /**
     * Show the form for editing the specified assignment.
     */
    public function edit(Assignment $assignment)
    {
        // Ensure user owns this assignment
        if ($assignment->user_id !== Auth::id()) {
            abort(403);
        }

        return view('assignments.edit', compact('assignment'));
    }

    /**
     * Update the specified assignment.
     */
    public function update(Request $request, Assignment $assignment)
    {
        // Ensure user owns this assignment
        if ($assignment->user_id !== Auth::id()) {
            abort(403);
        }

        try {
            $this->assignmentService->updateAssignment($assignment, $request->all());

            return redirect()
                ->route('assignments.show', $assignment)
                ->with('success', 'Assignment updated successfully!');
        } catch (ValidationException $e) {
            return back()
                ->withErrors($e->errors())
                ->withInput();
        }
    }

    /**
     * Mark assignment as completed.
     */
    public function complete(Assignment $assignment)
    {
        // Ensure user owns this assignment
        if ($assignment->user_id !== Auth::id()) {
            abort(403);
        }

        $this->assignmentService->completeAssignment($assignment);

        return redirect()
            ->route('weekly-planner')
            ->with('success', 'Assignment marked as completed!');
    }

    /**
     * Remove the specified assignment.
     * Note: Assignments should never be deleted per requirements, but we'll allow it for edge cases.
     */
    public function destroy(Assignment $assignment)
    {
        // Ensure user owns this assignment
        if ($assignment->user_id !== Auth::id()) {
            abort(403);
        }

        $assignment->delete();

        return redirect()
            ->route('assignments.index')
            ->with('success', 'Assignment deleted successfully!');
    }
}
