<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SystemSetting;
use App\Services\WhatsAppService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SettingsController extends Controller
{
    /**
     * Show system settings page.
     */
    public function index()
    {
        $settings = [
            'wasender_api_key' => SystemSetting::get('wasender_api_key', ''),
            'wasender_api_url' => SystemSetting::get('wasender_api_url', 'https://api.wasender.com'),
            'whatsapp_enabled' => SystemSetting::get('whatsapp_enabled', true),
            'trial_duration_days' => SystemSetting::get('trial_duration_days', 14),
            'default_quiet_hours_start' => SystemSetting::get('default_quiet_hours_start', '18:00'),
            'default_quiet_hours_end' => SystemSetting::get('default_quiet_hours_end', '08:00'),
        ];

        return view('admin.settings', compact('settings'));
    }

    /**
     * Update system settings.
     */
    public function update(Request $request)
    {
        $request->validate([
            'wasender_api_key' => ['nullable', 'string'],
            'wasender_api_url' => ['required', 'string', 'url'],
            'whatsapp_enabled' => ['boolean'],
            'trial_duration_days' => ['required', 'integer', 'min:1', 'max:365'],
            'default_quiet_hours_start' => ['required', 'string', 'regex:/^([0-1][0-9]|2[0-3]):[0-5][0-9]$/'],
            'default_quiet_hours_end' => ['required', 'string', 'regex:/^([0-1][0-9]|2[0-3]):[0-5][0-9]$/'],
        ]);

        SystemSetting::set('wasender_api_key', $request->wasender_api_key ?? '', 'string');
        SystemSetting::set('wasender_api_url', $request->wasender_api_url, 'string');
        SystemSetting::set('whatsapp_enabled', $request->has('whatsapp_enabled'), 'boolean');
        SystemSetting::set('trial_duration_days', $request->trial_duration_days, 'integer');
        SystemSetting::set('default_quiet_hours_start', $request->default_quiet_hours_start, 'string');
        SystemSetting::set('default_quiet_hours_end', $request->default_quiet_hours_end, 'string');

        // Also update config/services.php values if needed
        // This would require updating the config file or using config cache

        return redirect()
            ->route('admin.settings')
            ->with('success', 'Settings updated successfully!');
    }

    /**
     * Test WhatsApp sending.
     */
    public function testWhatsApp(Request $request)
    {
        $request->validate([
            'test_phone' => ['required', 'string'],
        ]);

        $user = Auth::user();
        $testPhone = $request->test_phone;

        // Temporarily set user's phone for testing
        $originalPhone = $user->phone;
        $user->phone = $testPhone;

        $whatsappService = new WhatsAppService();
        $message = "🧪 Test Message from Assignment Accountability System\n\n";
        $message .= "This is a test message to verify WhatsApp integration is working correctly.\n";
        $message .= "If you received this, your Wasender API is configured properly!";

        $result = $whatsappService->sendMessage($user, $message);

        // Restore original phone
        $user->phone = $originalPhone;

        if ($result) {
            return back()->with('success', 'Test message sent successfully! Check your WhatsApp.');
        } else {
            return back()->withErrors(['error' => 'Failed to send test message. Please check your API configuration and phone number.']);
        }
    }
}
