<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Assignment;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Show admin dashboard.
     */
    public function index()
    {
        $stats = [
            'total_users' => User::count(),
            'active_users' => User::whereIn('subscription_status', ['trial', 'paid'])->count(),
            'total_assignments' => Assignment::count(),
            'pending_assignments' => Assignment::pending()->count(),
            'completed_assignments' => Assignment::completed()->count(),
            'overdue_assignments' => Assignment::overdue()->count(),
            'trial_users' => User::where('subscription_status', 'trial')->count(),
            'paid_users' => User::where('subscription_status', 'paid')->count(),
            'free_users' => User::where('subscription_status', 'free')->count(),
        ];

        // Recent users
        $recentUsers = User::latest()->take(10)->get();

        // Recent assignments
        $recentAssignments = Assignment::with('user')->latest()->take(10)->get();

        // Users by subscription status
        $usersByStatus = User::select('subscription_status', DB::raw('count(*) as total'))
            ->groupBy('subscription_status')
            ->get();

        return view('admin.dashboard', compact('stats', 'recentUsers', 'recentAssignments', 'usersByStatus'));
    }

    /**
     * Show all users.
     */
    public function users()
    {
        $users = User::with('preferences')
            ->latest()
            ->paginate(20);

        return view('admin.users', compact('users'));
    }

    /**
     * Show user details.
     */
    public function showUser(User $user)
    {
        $user->load(['assignments', 'preferences', 'appNotifications']);
        
        return view('admin.user-details', compact('user'));
    }

    /**
     * Toggle admin status.
     */
    public function toggleAdmin(User $user)
    {
        $user->update(['is_admin' => !$user->is_admin]);
        
        return back()->with('success', 'Admin status updated successfully.');
    }
}
