<?php

namespace App\Console\Commands;

use App\Models\User;
use App\Models\UserPreference;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;

class CreateAdminUser extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'create:admin';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create an admin user account';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Creating Admin User...');
        $this->newLine();

        $name = $this->ask('Name');
        $email = $this->ask('Email');
        
        // Check if user already exists
        if (User::where('email', $email)->exists()) {
            if (!$this->confirm('User with this email already exists. Make them admin instead?')) {
                $this->info('Cancelled.');
                return;
            }
            
            $user = User::where('email', $email)->first();
            $user->update(['is_admin' => true]);
            $this->info("✓ User '{$user->name}' is now an admin!");
            return;
        }

        $password = $this->secret('Password');
        $passwordConfirmation = $this->secret('Confirm Password');

        if ($password !== $passwordConfirmation) {
            $this->error('Passwords do not match!');
            return;
        }

        $user = User::create([
            'name' => $name,
            'email' => $email,
            'password' => Hash::make($password),
            'subscription_status' => 'paid',
            'is_admin' => true,
            'trial_start_date' => Carbon::now(),
            'trial_expiry_date' => Carbon::now()->addDays(14),
        ]);

        // Create default preferences
        UserPreference::create([
            'user_id' => $user->id,
            'quiet_hours_start' => Carbon::now()->setTime(18, 0),
            'quiet_hours_end' => Carbon::now()->setTime(8, 0),
            'onboarding_completed' => true,
        ]);

        $this->newLine();
        $this->info('✓ Admin user created successfully!');
        $this->table(
            ['Field', 'Value'],
            [
                ['Name', $user->name],
                ['Email', $user->email],
                ['Admin', 'Yes'],
                ['Status', 'Paid'],
            ]
        );
        $this->newLine();
        $this->info('You can now login at: http://localhost/reminder/login');
    }
}
